'use client';

import { Modal } from "@/components/ui/modal";
import Button from "@/components/ui/button/Button";
import { CopyIcon } from "@/icons";
import { SERVICE_STATUSES } from "@/lib/prescription-statuses";
import Badge from "@/components/ui/badge/Badge";

interface Prescription {
  id: number;
  patient_first_name: string;
  patient_last_name: string;
  patient_national_id: string;
  patient_base_insurance_issuer: number;
  patient_supplementary_insurance_issuer: number;
  prescription_number: number;
  doctor_name: string;
  created_at: string;
  created_by_name: string;
  service_status: string;
  service_status_updated_at: string;
}

interface Insurance {
  id: number;
  name: string;
}

type BadgeColor = 
  | "primary"
  | "success"
  | "error"
  | "warning"
  | "info"
  | "light"
  | "dark";

interface PrescriptionDetailModalProps {
  isOpen: boolean;
  onClose: () => void;
  prescription: Prescription;
  baseInsurances: Insurance[];
  supplementaryInsurances: Insurance[];
  onCopyToClipboard: (text: string, fieldName: string) => void;
}

const PrescriptionDetailModal = ({
  isOpen,
  onClose,
  prescription,
  baseInsurances,
  supplementaryInsurances,
  onCopyToClipboard,
}: PrescriptionDetailModalProps) => {
  if (!isOpen) {
    return null;
  }

  const insuranceName = baseInsurances.find(i => i.id === prescription.patient_base_insurance_issuer)?.name || prescription.patient_base_insurance_issuer;
  const supplementaryInsuranceName = supplementaryInsurances.find(i => i.id === prescription.patient_supplementary_insurance_issuer)?.name || prescription.patient_supplementary_insurance_issuer;
  const statusColor = (SERVICE_STATUSES.find(s => s.value === prescription.service_status.toString())?.color || 'light') as BadgeColor;
  const statusDisplay = SERVICE_STATUSES.find(s => s.value === prescription.service_status.toString())?.label || prescription.service_status;

  return (
    <Modal
      isOpen={isOpen}
      onClose={onClose}
      className="max-w-[584px] p-5 lg:p-10"
    >
      <div className="text-gray-800 dark:text-gray-200">
        <h4 className="mb-6 text-lg font-medium text-gray-800 dark:text-white/90">
          جزییات نسخه
        </h4>

        <div className="grid grid-cols-1 gap-x-6 gap-y-5 sm:grid-cols-2">
          <div className="col-span-1 p-2 border border-gray-200 dark:border-gray-700 rounded-md">
            <p className="mb-1 text-xs text-gray-500 dark:text-gray-400">نام بیمار</p>
            <p className="font-medium">{`${prescription.patient_first_name} ${prescription.patient_last_name}`}</p>
          </div>

          <div className="col-span-1 p-2 border border-gray-200 dark:border-gray-700 rounded-md">
            <p className="mb-1 text-xs text-gray-500 dark:text-gray-400">نام پزشک</p>
            <p className="font-medium">{prescription.doctor_name}</p>
          </div>

          <div
            className="col-span-1 cursor-pointer p-2 border border-gray-200 dark:border-gray-700 rounded-md bg-gray-100 dark:bg-gray-800 hover:bg-gray-200 dark:hover:bg-gray-700 transition-colors"
            onClick={() =>
              onCopyToClipboard(
                prescription.patient_national_id,
                "کد ملی بیمار"
              )
            }
            title="کپی کردن"
          >
            <p className="mb-1 text-xs text-gray-500 dark:text-gray-400">کد ملی بیمار</p>
            <div className="flex items-center font-medium">
              <CopyIcon className="w-4 h-4" />
              <div className="mr-2">{prescription.patient_national_id}</div>
            </div>
          </div>

          <div
            className="col-span-1 cursor-pointer p-2 border border-gray-200 dark:border-gray-700 rounded-md bg-gray-100 dark:bg-gray-800 hover:bg-gray-200 dark:hover:bg-gray-700 transition-colors"
            onClick={() =>
              onCopyToClipboard(
                String(prescription.prescription_number),
                "کد پیگیری"
              )
            }
            title="کپی کردن"
          >
            <p className="mb-1 text-xs text-gray-500 dark:text-gray-400">کد پیگیری</p>
            <div className="flex items-center font-medium">
              <CopyIcon className="w-4 h-4" />
              <div className="mr-2">{prescription.prescription_number}</div>
            </div>
          </div>

          <div className="col-span-1 p-2 border border-gray-200 dark:border-gray-700 rounded-md">
            <p className="mb-1 text-xs text-gray-500 dark:text-gray-400">بیمه پایه</p>
            <p className="font-medium">{insuranceName}</p>
          </div>

          <div className="col-span-1 p-2 border border-gray-200 dark:border-gray-700 rounded-md">
            <p className="mb-1 text-xs text-gray-500 dark:text-gray-400">بیمه تکمیلی</p>
            <p className="font-medium">{supplementaryInsuranceName}</p>
          </div>

          <div className="col-span-1 p-2 border border-gray-200 dark:border-gray-700 rounded-md">
            <p className="mb-1 text-xs text-gray-500 dark:text-gray-400">وضعیت</p>
            <Badge color={statusColor}>
              {statusDisplay}
            </Badge>
          </div>
          <div className="col-span-1 p-2 border border-gray-200 dark:border-gray-700 rounded-md">
            <p className="mb-1 text-xs text-gray-500 dark:text-gray-400">ایجاد شده توسط</p>
            <p className="font-medium">{prescription.created_by_name}</p>
          </div>

          <div className="col-span-1 p-2 border border-gray-200 dark:border-gray-700 rounded-md">
            <p className="mb-1 text-xs text-gray-500 dark:text-gray-400">تاریخ ایجاد</p>
            <p className="font-medium">
              {new Date(prescription.created_at).toLocaleString("fa-IR")}
            </p>
          </div>

          {prescription.service_status_updated_at && (
            <div className="col-span-1 p-2 border border-gray-200 dark:border-gray-700 rounded-md">
              <p className="mb-1 text-xs text-gray-500 dark:text-gray-400">تاریخ بروزرسانی وضعیت</p>
              <p className="font-medium">
                {new Date(prescription.service_status_updated_at).toLocaleString("fa-IR")}
              </p>
            </div>
          )}
        </div>

        <div className="flex items-center justify-end w-full gap-3 mt-6">
          <Button size="sm" variant="outline" onClick={onClose}>
            بستن
          </Button>
        </div>
      </div>
    </Modal>
  );
};

export default PrescriptionDetailModal;
