"use client";

import { fetchWrapper } from "@/api/fetchWrapper";
import { BASE_URL } from "@/api/config";
import { useEffect, useState } from "react";
import { DocumentTextIcon, ClockIcon } from "@heroicons/react/24/outline";
import ComponentCard from "@/components/common/ComponentCard";

interface OverviewData {
    total_prescriptions: number;
    total_unchecked_prescriptions: number;
    total_prescriptions_today: number;
    total_unchecked_prescriptions_today: number;
}

interface StatCardProps {
  icon: React.ElementType;
  value: number;
  label: string;
}

const StatCard: React.FC<StatCardProps> = ({ icon: Icon, value, label }) => (
  <div className="rounded-sm bg-gray-100 dark:bg-gray-800 p-4">
    <div className="flex h-11.5 w-11.5 items-center justify-center rounded-full bg-meta-2 dark:bg-meta-4">
      <Icon className="w-6 h-6 text-primary dark:text-white" />
    </div>
    <div className="mt-4 flex items-end justify-between">
      <div>
        <h4 className="text-title-md font-bold text-black dark:text-white">
          {value}
        </h4>
        <span className="text-sm font-medium text-gray-600 dark:text-gray-300">{label}</span>
      </div>
    </div>
  </div>
);

const Overview = () => {
  const [data, setData] = useState<OverviewData | null>(null);
  const [error, setError] = useState<string | null>(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchData = async () => {
      try {
        const response = await fetchWrapper(`${BASE_URL}/api/dashboard/overview/`);
        if (response.ok) {
          const result = await response.json();
          setData(result);
        } else {
          setError("عدم امکان دریافت داده‌های کلی. لطفاً بعداً دوباره تلاش کنید.");
        }
      } catch {
        setError("خطایی در حین دریافت داده‌ها رخ داد.");
      } finally {
        setLoading(false);
      }
    };

    fetchData();
  }, []);

  if (loading) {
    return <div>Loading...</div>;
  }

  if (error) {
    return <div className="p-4 text-red-500 bg-red-100 rounded-md">{error}</div>;
  }

  return (
    <div className="space-y-6">
      <ComponentCard title="آمار امروز">
        <div className="grid grid-cols-1 gap-4 md:grid-cols-2 md:gap-6 2xl:gap-7.5">
          <StatCard
            icon={DocumentTextIcon}
            value={data?.total_prescriptions_today ?? 0}
            label="تعداد نسخه های امروز"
          />
          <StatCard
            icon={ClockIcon}
            value={data?.total_unchecked_prescriptions_today ?? 0}
            label="تعداد نسخه های بررسی نشده امروز"
          />
        </div>
      </ComponentCard>

      <ComponentCard title="آمار کل">
        <div className="grid grid-cols-1 gap-4 md:grid-cols-2 md:gap-6 2xl:gap-7.5">
          <StatCard
            icon={DocumentTextIcon}
            value={data?.total_prescriptions ?? 0}
            label="تعداد نسخه های کل"
          />
          <StatCard
            icon={ClockIcon}
            value={data?.total_unchecked_prescriptions ?? 0}
            label="تعداد نسخه های بررسی نشده کل"
          />
        </div>
      </ComponentCard>
    </div>
  );
};

export default Overview;
